<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/auth.php';
require_once __DIR__ . '/../middleware/jwtMiddleware.php';
require_once __DIR__ . '/../vendor/autoload.php';

use Firebase\JWT\JWT;

class AuthController {
    private $authModel;
    private $db;
    
    public function __construct() {
        $database = new Database();
        $db = $database->getConnection();
        $this->authModel = new Auth($db);
        error_log("✅ AuthController initialized");
    }

public function signup() {
    // Get raw JSON data
    $input = file_get_contents("php://input");
    error_log("📥 Signup request received: " . $input);
    
    $data = json_decode($input, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("❌ JSON decode error: " . json_last_error_msg());
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }

    // Validate required fields
    $required = ['name', 'email', 'password', 'confirm_password', 'nickname', 'gender', 'age', 'location', 'preference'];
    foreach ($required as $field) {
        if (empty($data[$field])) {
            error_log("❌ Missing required field: $field");
            http_response_code(400);
            echo json_encode(['error' => "Field '$field' is required"]);
            return;
        }
    }

    // Validate email format
    if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
        error_log("❌ Invalid email format: " . $data['email']);
        http_response_code(400);
        echo json_encode(['error' => 'Invalid email format']);
        return;
    }

    // Check if email exists
    $existingUser = $this->authModel->emailExists($data['email']);
    if ($existingUser) {
        error_log("❌ Email already registered: " . $data['email']);
        http_response_code(400);
        echo json_encode(['error' => 'Email already registered']);
        return;
    }

    // Password validation
    $password = $data['password'];
    $confirmPassword = $data['confirm_password'];
    
    // Check password match
    if ($password !== $confirmPassword) {
        error_log("❌ Passwords do not match");
        http_response_code(400);
        echo json_encode(['error' => 'Passwords do not match']);
        return;
    }

    // Password strength validation
    $passwordErrors = $this->validatePasswordStrength($password);
    if (!empty($passwordErrors)) {
        error_log("❌ Weak password: " . implode(', ', $passwordErrors));
        http_response_code(400);
        echo json_encode([
            'error' => 'Weak password',
            'message' => implode(' ', $passwordErrors)
        ]);
        return;
    }

    // Validate age
    $age = (int)$data['age'];
    if ($age < 18 || $age > 100) {
        error_log("❌ Invalid age: $age");
        http_response_code(400);
        echo json_encode(['error' => 'Age must be between 18 and 100']);
        return;
    }

    // Validate gender
    $validGenders = ['male', 'female'];
    if (!in_array(strtolower($data['gender']), $validGenders)) {
        error_log("❌ Invalid gender: " . $data['gender']);
        http_response_code(400);
        echo json_encode(['error' => 'Gender must be Male or Female']);
        return;
    }

    // Validate preference
    $validPreferences = ['men', 'women', 'both'];
    if (!in_array(strtolower($data['preference']), $validPreferences)) {
        error_log("❌ Invalid preference: " . $data['preference']);
        http_response_code(400);
        echo json_encode(['error' => 'Preference must be Men, Women, or Both']);
        return;
    }

    // Get latitude and longitude from request (optional)
    $latitude = isset($data['latitude']) ? (float)$data['latitude'] : null;
    $longitude = isset($data['longitude']) ? (float)$data['longitude'] : null;
    
    error_log("📍 User coordinates - Lat: $latitude, Lng: $longitude");

    // Create user with all fields including coordinates
    $userId = $this->authModel->createUser(
        $data['name'], 
        $data['email'], 
        $password, // Now using user-entered password
        null, // googleId
        null, // avatarUrl
        $data['nickname'],
        $data['gender'],
        $age,
        $data['location'],
        $data['bio'] ?? '',
        $data['preference'],
        $latitude,
        $longitude
    );

    if ($userId) {
        error_log("✅ User created successfully with ID: $userId");
        
        // Return success with user ID
        http_response_code(201);
        echo json_encode([
            'success' => true,
            'message' => 'User created successfully',
            'user_id' => $userId
        ]);
    } else {
        error_log("❌ User creation failed in database");
        http_response_code(500);
        echo json_encode(['error' => 'User creation failed. Please try again.']);
    }
}

private function validatePasswordStrength($password) {
    $errors = [];
    
    // Minimum length
    if (strlen($password) < 8) {
        $errors[] = 'Password must be at least 8 characters long.';
    }
    
    // Check for uppercase
    if (!preg_match('/[A-Z]/', $password)) {
        $errors[] = 'Password must contain at least one uppercase letter.';
    }
    
    // Check for lowercase
    if (!preg_match('/[a-z]/', $password)) {
        $errors[] = 'Password must contain at least one lowercase letter.';
    }
    
    // Check for numbers
    if (!preg_match('/\d/', $password)) {
        $errors[] = 'Password must contain at least one number.';
    }
    
    // Optional: Check for special characters
    // if (!preg_match('/[!@#$%^&*()_+\-=\[\]{};\':"\\|,.<>\/?]/', $password)) {
    //     $errors[] = 'Password must contain at least one special character.';
    // }
    
    return $errors;
}   
 
    public function login() {
        $input = file_get_contents("php://input");
        error_log("📥 Login request received: " . $input);
        
        $data = json_decode($input, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON data']);
            return;
        }

        if (empty($data['email']) || empty($data['password'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Email and password required']);
            return;
        }

        $user = $this->authModel->emailExists($data['email']);
        
        if ($user && password_verify($data['password'], $user['password'])) {
            error_log("✅ Login successful for user: " . $user['email']);
            
            // Generate JWT
            $secretKey = $_ENV['JWT_SECRET'] ?? getenv('JWT_SECRET');
            
            if (!$secretKey) {
                error_log("❌ JWT_SECRET not found");
                $secretKey = 'sakssam@4537saka';
            }
            
            $expiry = time() + ((int)($_ENV['JWT_EXPIRY'] ?? getenv('JWT_EXPIRY') ?: 86400));
            
            $payload = [
                'iss' => $_ENV['APP_BASE_URL'] ?? 'http://localhost',
                'aud' => $_ENV['FRONTEND_URL'] ?? 'http://localhost:3000',
                'iat' => time(),
                'exp' => $expiry,
                'user_id' => $user['id'],
                'email' => $user['email']
            ];
            
            try {
                $jwt = JWT::encode($payload, $secretKey, 'HS256');
                error_log("✅ JWT created successfully for user ID: " . $user['id']);
            } catch (Exception $e) {
                error_log("❌ JWT creation failed: " . $e->getMessage());
                http_response_code(500);
                echo json_encode(['error' => 'Authentication system error']);
                return;
            }
            
            // Remove password from user data
            unset($user['password']);
            
            // Return success response
            echo json_encode([
                'success' => true,
                'message' => 'Login successful',
                'token' => $jwt,
                'user' => $user,
                'expires_in' => $expiry
            ]);
        } else {
            error_log("❌ Login failed for email: " . $data['email']);
            http_response_code(401);
            echo json_encode(['error' => 'Invalid email or password']);
        }
    }
    
    public function profile() {
        $userId = jwtMiddleware();
        
        $user = $this->authModel->getUserProfile($userId);
        
        if ($user) {
            echo json_encode([
                'success' => true,
                'user' => $user
            ]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'User not found']);
        }
    }

    public function logout() {
        echo json_encode([
            'success' => true,
            'message' => 'Logged out successfully'
        ]);
    }
}
?>